import { wayfinder } from '@laravel/vite-plugin-wayfinder';
import tailwindcss from '@tailwindcss/vite';
import vue from '@vitejs/plugin-vue';
import { readFileSync } from 'fs';
import laravel from 'laravel-vite-plugin';
import { resolve } from 'path';
import { defineConfig } from 'vite';

// Lê o .env do Laravel para obter o ASSET_URL
function loadEnv(): Record<string, string> {
    try {
        const envPath = resolve(process.cwd(), '.env');
        const envContent = readFileSync(envPath, 'utf-8');
        const env: Record<string, string> = {};

        envContent.split('\n').forEach((line) => {
            const trimmedLine = line.trim();
            if (trimmedLine && !trimmedLine.startsWith('#')) {
                const [key, ...valueParts] = trimmedLine.split('=');
                if (key && valueParts.length > 0) {
                    const value = valueParts.join('=').trim();
                    // Remove aspas se existirem
                    env[key.trim()] = value.replace(/^["']|["']$/g, '');
                }
            }
        });

        return env;
    } catch (error) {
        return {};
    }
}

// Determina o base path baseado no ASSET_URL ou VITE_BASE_PATH
// VITE_BASE_PATH tem prioridade e é específico para o Vite
function getBasePath(): string {
    const env = loadEnv();

    // Primeiro tenta VITE_BASE_PATH (específico para Vite)
    const viteBasePath = process.env.VITE_BASE_PATH || env.VITE_BASE_PATH;
    if (viteBasePath) {
        const path = viteBasePath.startsWith('/')
            ? viteBasePath
            : '/' + viteBasePath;
        return path.endsWith('/') ? path + 'build/' : path + '/build/';
    }

    // Depois tenta ASSET_URL (compatível com Laravel)
    const assetUrl = process.env.ASSET_URL || env.ASSET_URL;
    if (assetUrl && assetUrl.trim() !== '') {
        // Se ASSET_URL está definido, adiciona /build ao final
        const path = assetUrl.startsWith('/') ? assetUrl : '/' + assetUrl;
        return path.endsWith('/') ? path + 'build/' : path + '/build/';
    }

    // Padrão: /build/
    return '/build/';
}

export default defineConfig({
    base: getBasePath(),
    plugins: [
        laravel({
            input: ['resources/js/app.ts'],
            ssr: 'resources/js/ssr.ts',
            refresh: true,
        }),
        tailwindcss(),
        wayfinder({
            formVariants: true,
        }),
        vue({
            template: {
                transformAssetUrls: {
                    base: null,
                    includeAbsolute: false,
                },
            },
        }),
    ],
    server: {
        host: '0.0.0.0',
        port: 5173,
        strictPort: true,
        hmr: {
            host: 'localhost',
        },
    },
});
